import sys
import os
import asyncio
import threading
from PyQt5.QtWidgets import QDialog, QVBoxLayout, QPlainTextEdit, QWidget, QLineEdit, QPushButton, QLabel, QMessageBox, QApplication
from PyQt5.QtCore import QTimer
import anymc
import io
import argparse

class LogDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Launching Minecraft...")
        self.setFixedSize(600, 350)

        layout = QVBoxLayout()
        self.log_box = QPlainTextEdit()
        self.log_box.setReadOnly(True)
        layout.addWidget(self.log_box)
        self.setLayout(layout)

        if not debug:
            # Redirect stdout/stderr to log_box
            self.stdout = sys.stdout
            print(sys.stdout)
            print(sys.stderr)
            self.stderr = sys.stderr
            sys.stdout = self.stream = StreamRedirector(self.append_log)
            sys.stderr = self.stream

            # Polling updates
            self.timer = QTimer()
            self.timer.timeout.connect(self.stream.flush)
            self.timer.start(100)

        

    def append_log(self, text):
        self.log_box.appendPlainText(text)

    def closeEvent(self, event):
        # Restore stdout/stderr
        if not debug:
            sys.stdout = self.stdout
            sys.stderr = self.stderr
            super().closeEvent(event)

class StreamRedirector(io.StringIO):
    def __init__(self, callback):
        super().__init__()
        self.callback = callback
        self.buffer = ""

    def write(self, msg):
        self.buffer += msg
        if "\n" in self.buffer:
            lines = self.buffer.splitlines(keepends=True)
            for line in lines:
                if line.endswith("\n"):
                    self.callback(line.rstrip())
            self.buffer = ""

    def flush(self):
        if self.buffer:
            self.callback(self.buffer.rstrip())
            self.buffer = ""

class AnyMCGui(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("AnyMC Launcher")

        layout = QVBoxLayout()

        self.version_input = QLineEdit()
        self.version_input.setPlaceholderText("e.g. 1.21.5")

        self.username_input = QLineEdit()
        self.username_input.setPlaceholderText("e.g. DCreator112")

        self.launch_btn = QPushButton("Launch! 🚀")
        self.launch_btn.clicked.connect(self.launch_minecraft)

        layout.addWidget(QLabel("Version:"))
        layout.addWidget(self.version_input)
        layout.addWidget(QLabel("Username:"))
        layout.addWidget(self.username_input)
        layout.addWidget(self.launch_btn)

        self.setLayout(layout)

    def launch_minecraft(self):
        ver = self.version_input.text().strip()
        name = self.username_input.text().strip()

        if not ver or not name:
            QMessageBox.warning(self, "Missing Info", "Please enter both version and username.")
            return

        # Set AnyMC global config
        anymc.version = ver
        anymc.username = name
        anymc.base_dir = os.path.abspath(f"./anymc/{anymc.loader}-{ver}")
        anymc.libraries_dir = os.path.join(anymc.base_dir, "libraries")
        anymc.versions_dir = os.path.join(anymc.base_dir, "versions")
        anymc.game_dir = anymc.base_dir
        anymc.assets_dir = os.path.join(anymc.base_dir, "assets")
        anymc.debug = debug
        anymc.offline = offline

        # Show log window
        self.log_window = LogDialog(self)
        self.log_window.show()

        # Run AnyMC launcher in separate thread (non-blocking)
        threading.Thread(target=self.run_anymc).start()

    def run_anymc(self):
        asyncio.run(anymc.main())

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="GUI launcher for AnyMC")
    parser.add_argument("-d", "--debug", action="store_true", help="Enable debug mode")
    parser.add_argument("-o", "--offline", action="store_true", help="Do not try to download Minecraft")
    args = parser.parse_args()
    debug = args.debug
    offline = args.offline
    app = QApplication(sys.argv)
    gui = AnyMCGui()
    gui.resize(200, 225)
    gui.show()
    sys.exit(app.exec_())